
#ifndef PolymerSystem_hpp
#define PolymerSystem_hpp

#include <random>
#include <array>
#include <vector>
#include <iostream>
#include <stdlib.h>
#include <math.h>
#include <thread>
#include <mutex>




// set number of species everywhere and evaluate calculations at compiletime
constexpr int number_of_species=100;

//array for int numbers associated with left and right end
typedef std::array<std::array<long, number_of_species>, number_of_species> all_species_2D_array;
//same for doubles
typedef std::array<std::array<double, number_of_species>, number_of_species> all_species_2D_array_double;

/// class for complete system
class Complete_System{
    
public:

    /// Constructor
    /// @param critical_ring_size critical ring size
    /// @param initial_particales initial number of particles in the active state
    /// @param inactive_particles initial number of particles in the inactive state
    /// @param activation_rate activation rate
    /// @param dimerization_rate dimerization rate
    /// @param decay_rate decay rate of small cluster
    /// @param seed random number seed (has nothing to do with particle seeds!)
    Complete_System(const int critical_ring_size,const long initial_particales,const long inactive_particles,const double activation_rate,const double dimerization_rate,const double decay_rate,unsigned seed);
    
    /// Copy constructor
    /// @param other other system
    /// @param seed random number seed for different time evolution (has nothing to do with particle seeds!)
    Complete_System(Complete_System* other,unsigned seed);
    
    //helper functions for creating and removing unfinished rings
    /// Combines two unfinished rings
    /// @param left_end_1 left end of first ring
    /// @param right_end_1 right end of first ring
    /// @param left_end_2  left end of sceond ring
    /// @param right_end_2 right end of sceond ring
    void combine_rings(const int left_end_1,const int right_end_1, const int left_end_2,const int right_end_2);
    /// removes a ring from the system
    /// @param left_end left end
    /// @param right_end right end
    void remove_ring(const int left_end,const int right_end);
    /// Creates a ring
    /// @param left_end left end
    /// @param right_end right end
    void create_ring(const int left_end,const int right_end);
    
    //helper functions for decay event of a ring that is below the critical size
    /// Return all monomers of a ring
    /// @param left_site left end
    /// @param right_site right end
    void decompose_ring(const int left_site,const int right_site);
    /// Removes and decomposes a ring
    /// @param left_end left end
    /// @param right_end right end
    void destroy_ring(const int left_end,const int right_end);
    /// Chooses a random ring to destroy
    void destroy_small_ring_event();
    
    //helper functions to determine which particles react in a gillespie step
    /// Get the left end of the first reacting ring or other event type like decay or activation
    ///@return index of species or event type
    const int get_first_reacting_component_left_end();
    /// Get the right end of the first reacting ring
    /// @param left_end left and that was chosen in the previous step
    ///@return right end that was chosen
    const int get_first_reacting_component_right_end(const int left_end);
    /// Get the left end of the second reacting ring
    /// @param left_end_1 left end of ring one
    /// @param right_end_1 right end of ring two
    /// @param right_end_2 right end of second ring (given by binding conditions)
    ///@return index of left end species
    const int get_second_reacting_component_left_end(const int left_end_1,const int right_end_1,const int right_end_2);
    
    /// activate particle
    /// @param species particle species
    void activate_particle(const int species);
    
    //substeps for a gillespie iteration step
    /// make rate for next iteration step
    void make_rates();
    /// choos event for next iteration step
    void choose_event();
    
    /// perform single iteration step
    ///@return terminate simulation?
    bool iteration_step();

    /// perform a simulation with the parameters speciefied in the constructor
    void simulate();
    //write the results to an extern array for further processing
    
    /// Writes final simulation results
    /// @param yield pointer to which to return the yield
    void write_final_results( double* const yield);
    
    /// Perform simulation of a copy of the system
    /// /// @param seed random number seed (has nothing to do with particle seeds!)
    /// @param yield pointer to which to return the yield
    void operator()(const unsigned seed,double* const yield);
    
    /// Perform simulation of an ensemble of systems
    /// @param number_of_ensembles number of ensembles
    /// @param yield pointer to which to return the yield
    void GetEnsAv(const int number_of_ensembles,double* const yield);
    
    
private:
    //random number engine 32bit MT
    std::mt19937 rng_engine;
    //particles inactive at the beginning of the simulation
    std::array<long, number_of_species> _inactive_particles;
    //all particles present in the simulation. ordered by [left_end][right_end]
    all_species_2D_array _ring_structures;
    //int number that speciefies to how binding possibilites a species has
    std::array<long,number_of_species> _all_binding_combinations_of_species;
    //number of particles that can bind to a ring segment of type [left_end][right_end]
    all_species_2D_array _possible_bindings_2D;
    const double activation_rate_;
    const double decay_rate_;
    const double dimerization_rate_;
    //accumulated rates for the gillespie step:
    //[0 - number_of_species-1]: particle attaches to another one
    //[number_of_species]: to small seed decays
    //[number_ofspecies+1 - 2*number_of_species]: particle gets activated
    std::array<double,2*number_of_species+1>_rates;
    //bools to keep the simulation from evaluating the particle activation probability after all particles are activated
    bool _particles_left_to_activate;
    bool _deactivate_last_part_of_rates;
    //if all particles are activated not all entries of _rates have to be considered
    int _rate_counter;
    //size below which ring structures can decay
    int _critical_ring_size;
    // number of rings finished (evaluated at the end of the simulation)
    long _finished_rings;
    // number of stable structures that are not finished
    long _undesired_structures;
    // number of rings below the critical ring size
    long _destructable_rings;
    // sum of all particles at the beginning of the simulation
    long _total_number_of_components;
    // total number of inactive particles
    long _number_of_inactive_particles;
    //current simulation time
    double _time;
};

#endif /* PolymerSystem_hpp */
